/****************************************************************************
*
*    Copyright (c) 2005 - 2014 by Vivante Corp.  All rights reserved.
*
*    The material in this file is confidential and contains trade secrets
*    of Vivante Corporation. This is proprietary information owned by
*    Vivante Corporation. No part of this work may be disclosed,
*    reproduced, copied, transmitted, or used in any way for any purpose,
*    without the express written permission of Vivante Corporation.
*
*****************************************************************************/


#ifndef _tiny_ui_kernel_h_
#define _tiny_ui_kernel_h_

#include "tiny_ui_error.h"

#ifdef __cplusplus
extern "C" {
#endif

// Alignment macro.
#define TINY_UI_ALIGN(number, alignment)    (((number) + ((alignment) - 1)) & ~((alignment) - 1))

typedef enum tiny_ui_kernel_counter
{
    /* Dont't touch the counter. */
    TINY_UI_NONE,

    /* Turn the counter on. */
    TINY_UI_ON,

    /* Turn the counter off. */
    TINY_UI_OFF,

    /* Query the counter and reset its values. */
    TINY_UI_QUERY,
}
tiny_ui_kernel_counter_t;

typedef enum tiny_ui_kernel_command
{
    /* Initialize the GPU. */
    TINY_UI_INITIALIZE,

    /* Terminate the GPU. */
    TINY_UI_TERMINATE,

    /* Allocate memory. */
    TINY_UI_ALLOCATE,

    /* Free memory. */
    TINY_UI_FREE,

    /* Submit a command buffer to the GPU. */
    TINY_UI_SUBMIT,

    /* Wait for the GPU to be completed. */
    TINY_UI_WAIT,

    /* Reset the GPU. */
    TINY_UI_RESET,

    /* Debug commands. */
    TINY_UI_DEBUG,

    /* Map memory. */
    TINY_UI_MAP,

    /* Unmap memory. */
    TINY_UI_UNMAP,
}
tiny_ui_kernel_command_t;

/* Context structure. */
typedef struct tiny_ui_kernel_context tiny_ui_kernel_context_t;

typedef struct tiny_ui_capabilities
{
    uint32_t tiled : 2;
    uint32_t l2_cache : 1;
    uint32_t l8 : 1;
    uint32_t border_culling : 1;
}
tiny_ui_capabilities_t;

typedef struct tiny_ui_kernel_initialize
{
    /* Command buffer size. */
    uint32_t command_buffer_size;

    /* Tessellation buffer width. */
    int32_t tessellation_width;

    /* Tessellation buffer height. */
    int32_t tessellation_height;

    /* OUTPUT */

    /* Context pointer. */
    tiny_ui_kernel_context_t * context;

    /* Capabilities. */
    tiny_ui_capabilities_t capabilities;

    /* Allocated command buffer. */
    void * command_buffer;

    /* GPU address for command buffer. */
    uint32_t command_buffer_gpu;

    /* GPU addresses for tesselation buffers. */
    uint32_t tessellation_buffer_gpu[3];

    /* Size of each level of the tesselation buffer. */
    uint32_t tessellation_buffer_size[3];

    /* Stride of the tessellation buffer. */
    uint32_t tessellation_stride;

    /* Width and height of tessellation buffer. */
    uint32_t tessellation_width_height;
}
tiny_ui_kernel_initialize_t;

typedef struct tiny_ui_kernel_terminate
{
    /* Context to terminate. */
    tiny_ui_kernel_context_t * context;
}
tiny_ui_kernel_terminate_t;

typedef struct tiny_ui_kernel_allocate
{
    /* Number of bytes to allocate. */
    uint32_t bytes;

    /* Flag to indicate whether the allocated memory is contiguous or not. */
    int32_t contiguous;

    /* OUTPUT */

    /* Memory handle. */
    void * memory_handle;

    /* Allocated memory. */
    void * memory;

    /* GPU address of allocated memory. */
    uint32_t memory_gpu;
}
tiny_ui_kernel_allocate_t;

typedef struct tiny_ui_kernel_free
{
    /* Memory handle to free. */
    void * memory_handle;
}
tiny_ui_kernel_free_t;

typedef struct tiny_ui_kernel_submit
{
    /* Context to submit to. */
    tiny_ui_kernel_context_t * context;

    /* Pointer to command buffer. */
    void * commands;

    /* Number of bytes in command buffer. */
    uint32_t command_size;
}
tiny_ui_kernel_submit_t;

#define TINY_UI_INFINITE    0xFFFFFFFF

typedef struct tiny_ui_kernel_wait
{
    /* Context to wait for. */
    tiny_ui_kernel_context_t * context;

    /* Timeout in milliseconds. */
    uint32_t timeout_ms;
}
tiny_ui_kernel_wait_t;

typedef struct tiny_ui_kernel_reset
{
    /* Context to reset. */
    tiny_ui_kernel_context_t * context;
}
tiny_ui_kernel_reset_t;

typedef struct tiny_ui_kernel_debug
{
    /* Context to debug. */
    tiny_ui_kernel_context_t * context;

    /* Bandwidth counter enabler. */
    tiny_ui_kernel_counter_t bandwidth_counter;

    /* Pixel counter enabler. */
    tiny_ui_kernel_counter_t pixel_counters;

    /* OUTPUT */

    /* Bandwidth counters:
     *  [0] - burst of 8.
     *  [1] - burst of 16.
     *  [2] - burst of 32.
     *  [3] - burst of 64.
     */
    uint32_t bandwidth[4];

    /* Pixel counters:.
     *  [0] - Number of tessellated pixels.
     *  [1] - Number of imaged pixels.
     *  [2] - Number of rendered pixels.
     */
    uint32_t pixels[3];
}
tiny_ui_kernel_debug_t;

typedef struct tiny_ui_kernel_map
{
    /* Number of bytes to map. */
    uint32_t bytes;

    /* Logical memory address or NULL. */
    void * logical;

    /* Physical memory address or 0. */
    uint32_t physical;

    /* OUTPUT */

    /* Memory handle for mapped memory. */
    void * memory_handle;

    /* GPU address of mapped memory. */
    uint32_t memory_gpu;
}
tiny_ui_kernel_map_t;

typedef struct tiny_ui_kernel_unmap
{
    /* Memory handle to unmap. */
    void * memory_handle;
}
tiny_ui_kernel_unmap_t;

/* This is the function to call from the tinyUI driver to interface with the GPU. */
tiny_ui_error_t tiny_ui_kernel(tiny_ui_kernel_command_t command, void * data);

#ifdef __cplusplus
}
#endif
#endif /* _tiny_ui_kernel_h_ */
